jQuery(document).ready(function ($) {
    let imageQueue = [];
    let totalImages = 0;
    let processedImages = 0;
    let successfulConversions = 0;
    let failedConversions = 0;

    const initialState = $('#ewo-initial-state');
    const processingState = $('#ewo-processing-state');
    const completeState = $('#ewo-complete-state');
    const logContainer = $('.ewo-log-box-container');
    const logBox = $('#ewo-log');
    const qualitySlider = $('#ewo-quality-slider');
    const qualityValue = $('#ewo-quality-value');

    // Lógica de Pestañas
    $('.ewo-tab-button').on('click', function() {
        const tabId = $(this).data('tab');
        $('.ewo-tab-button').removeClass('active');
        $(this).addClass('active');
        $('.ewo-tab-content').removeClass('active');
        $('#ewo-tab-' + tabId).addClass('active');
    });

    // Actualizar el valor del slider en la UI
    qualitySlider.on('input', function() {
        qualityValue.text($(this).val());
    });

    // Iniciar el escaneo
    $('#ewo-start-scan-btn').on('click', function () {
        const btn = $(this);
        btn.prop('disabled', true).text('Escaneando...');
        logContainer.show();
        log('Iniciando escaneo de la biblioteca de medios...');

        $.post(ewo_ajax.ajax_url, {
            action: 'ewo_scan_images',
            nonce: ewo_ajax.nonce,
        })
        .done(function (response) {
            if (response.success) {
                imageQueue = response.data;
                totalImages = imageQueue.length;
                log(`Escaneo completado. Se encontraron ${totalImages} imágenes para procesar.`);
                
                if (totalImages > 0) {
                    initialState.hide();
                    processingState.show();
                    $('#ewo-total-count').text(totalImages);
                    processQueue();
                } else {
                    log('¡Felicidades! Todas las imágenes ya están optimizadas.');
                    btn.prop('disabled', false).text('Escanear Biblioteca de Medios');
                }
            } else {
                log('Error: ' + response.data.message, 'error');
                btn.prop('disabled', false).text('Escanear Biblioteca de Medios');
            }
        })
        .fail(function () {
            log('Error: No se pudo comunicar con el servidor.', 'error');
            btn.prop('disabled', false).text('Escanear Biblioteca de Medios');
        });
    });
    
    // Procesar la cola de imágenes
    function processQueue() {
        if (imageQueue.length === 0) {
            processingState.hide();
            completeState.show();
            $('#ewo-results-summary').html(`
                <p style="color: #cbd5e1; font-size: 16px;">Proceso finalizado.</p>
                <ul style="list-style: none; padding: 0; margin-top: 10px;">
                    <li style="color: #4ade80;">Conversiones exitosas: ${successfulConversions}</li>
                    <li style="color: #f87171;">Conversiones fallidas: ${failedConversions}</li>
                </ul>
            `);
            log('Proceso de optimización completado.');
            return;
        }

        const image = imageQueue.shift();
        processedImages++;
        
        updateProgressBar();
        $('#ewo-current-file').text(`Procesando: ${image.file}`);
        log(`(${processedImages}/${totalImages}) Procesando imagen ID: ${image.id} (${image.file})`);

        const currentQuality = qualitySlider.val();

        $.post(ewo_ajax.ajax_url, {
            action: 'ewo_convert_image',
            nonce: ewo_ajax.nonce,
            attachment_id: image.id,
            quality: currentQuality
        })
        .done(function (response) {
            if (response.success) {
                successfulConversions++;
                log(`ID ${image.id}: ${response.data.message}`, 'success');
            } else {
                failedConversions++;
                log(`ID ${image.id}: ${response.data.message}`, 'error');
            }
        })
        .fail(function () {
            failedConversions++;
            log(`ID ${image.id}: Falló la comunicación con el servidor.`, 'error');
        })
        .always(function () {
            setTimeout(processQueue, 200);
        });
    }

    // Lógica para el botón de restaurar
    $('#ewo-revert-all-btn').on('click', function() {
        if (!confirm(ewo_ajax.confirm_revert)) {
            return;
        }
        const btn = $(this);
        const statusDiv = $('#ewo-revert-status');
        
        btn.prop('disabled', true).text('Restaurando...');
        statusDiv.text('El proceso de restauración ha comenzado. Esto puede tardar varios minutos...').show();

        $.post(ewo_ajax.ajax_url, {
            action: 'ewo_revert_all_images',
            nonce: ewo_ajax.nonce,
        })
        .done(function(response) {
            if (response.success) {
                statusDiv.css('color', '#4ade80').text(response.data.message);
            } else {
                statusDiv.css('color', '#f87171').text('Error: ' + response.data.message);
            }
        })
        .fail(function() {
            statusDiv.css('color', '#f87171').text('Error: No se pudo comunicar con el servidor.');
        })
        .always(function() {
            btn.prop('disabled', false).text('Restaurar Todas las Imágenes');
        });
    });

    // Actualizar la barra de progreso
    function updateProgressBar() {
        const percentage = totalImages > 0 ? (processedImages / totalImages) * 100 : 0;
        $('#ewo-progress-bar').css('width', percentage + '%');
        $('#ewo-processed-count').text(processedImages);
    }

    // Reiniciar la interfaz
    $('#ewo-restart-btn').on('click', function() {
        imageQueue = [];
        totalImages = 0;
        processedImages = 0;
        successfulConversions = 0;
        failedConversions = 0;
        
        completeState.hide();
        logContainer.hide();
        initialState.show();
        $('#ewo-start-scan-btn').prop('disabled', false).text('Escanear Biblioteca de Medios');
        logBox.html('<p>Esperando para iniciar el proceso...</p>');
        updateProgressBar();
    });

    // Función para añadir mensajes al log
    function log(message, type = 'info') {
        const colorClass = type === 'error' ? 'color: #f87171;' : (type === 'success' ? 'color: #4ade80;' : '');
        logBox.append(`<p style="${colorClass}">${message}</p>`);
        logBox.scrollTop(logBox[0].scrollHeight);
    }
});
