<?php
if (!defined('ABSPATH')) {
    exit;
}

// Registrar los endpoints de AJAX
add_action('wp_ajax_ewo_scan_images', 'ewo_scan_images_callback');
add_action('wp_ajax_ewo_convert_image', 'ewo_convert_image_callback');
add_action('wp_ajax_ewo_revert_all_images', 'ewo_revert_all_images_callback');

/**
 * Escanea la biblioteca de medios en busca de imágenes para convertir.
 */
function ewo_scan_images_callback() {
    check_ajax_referer('ewo_ajax_nonce', 'nonce');

    if (!function_exists('imagewebp')) {
        wp_send_json_error(['message' => 'El servidor no soporta la conversión a WebP (librería GD no disponible o sin soporte WebP).']);
    }

    $images_to_process = ewo_get_images_to_process();
    wp_send_json_success($images_to_process);
}

/**
 * Convierte una sola imagen a WebP.
 */
function ewo_convert_image_callback() {
    check_ajax_referer('ewo_ajax_nonce', 'nonce');

    $attachment_id = isset($_POST['attachment_id']) ? intval($_POST['attachment_id']) : 0;
    $quality = isset($_POST['quality']) ? intval($_POST['quality']) : 85;

    $result = ewo_perform_conversion($attachment_id, $quality);

    if (is_wp_error($result)) {
        wp_send_json_error(['message' => $result->get_error_message()]);
    } else {
        wp_send_json_success(['message' => 'Convertido con éxito.']);
    }
}

/**
 * Restaura todas las imágenes a su estado original.
 */
function ewo_revert_all_images_callback() {
    check_ajax_referer('ewo_ajax_nonce', 'nonce');
    
    global $wpdb;
    $args = [
        'post_type'      => 'attachment',
        'post_status'    => 'inherit',
        'posts_per_page' => -1,
        'meta_query'     => [
            [
                'key'     => '_ewo_is_converted',
                'value'   => '1',
                'compare' => '=',
            ],
        ],
    ];

    $query = new WP_Query($args);
    $upload_dir = wp_get_upload_dir();
    $reverted_count = 0;

    foreach ($query->posts as $attachment) {
        $metadata = wp_get_attachment_metadata($attachment->ID);
        if (!$metadata || !isset($metadata['file'])) continue;

        // 1. Restaurar URLs en la base de datos
        $image_sizes = isset($metadata['sizes']) ? $metadata['sizes'] : [];
        $image_sizes['full'] = ['file' => basename($metadata['file'])];

        foreach ($image_sizes as $size_data) {
            $original_url = $upload_dir['baseurl'] . '/' . (dirname($metadata['file']) !== '.' ? dirname($metadata['file']) . '/' : '') . $size_data['file'];
            $webp_url = $original_url . '.webp';
            
            $wpdb->query($wpdb->prepare("UPDATE {$wpdb->posts} SET post_content = REPLACE(post_content, %s, %s)", $webp_url, $original_url));
            $wpdb->query($wpdb->prepare("UPDATE {$wpdb->postmeta} SET meta_value = REPLACE(meta_value, %s, %s)", $webp_url, $original_url));

            // 2. Eliminar el archivo .webp
            $webp_path = $upload_dir['basedir'] . '/' . (dirname($metadata['file']) !== '.' ? dirname($metadata['file']) . '/' : '') . $size_data['file'] . '.webp';
            if (file_exists($webp_path)) {
                @unlink($webp_path);
            }
        }

        // 3. Limpiar metadatos del plugin
        delete_post_meta($attachment->ID, '_ewo_is_converted');
        delete_post_meta($attachment->ID, '_ewo_conversion_timestamp');
        delete_post_meta($attachment->ID, '_ewo_original_deleted');
        $reverted_count++;
    }

    wp_send_json_success(['message' => sprintf('%d imágenes han sido restauradas a su formato original.', $reverted_count)]);
}
