<?php
if (!defined('ABSPATH')) {
    exit;
}

// Registrar las acciones para los hooks de cron
add_action('ewo_daily_cleanup_hook', 'ewo_scheduled_cleanup');
add_action('ewo_weekly_scan_hook', 'ewo_scheduled_scan');

/**
 * Tarea diaria: elimina los archivos originales de imágenes convertidas hace más de 2 días.
 */
function ewo_scheduled_cleanup() {
    $threshold = time() - (2 * DAY_IN_SECONDS);
    $args = [
        'post_type'      => 'attachment',
        'post_status'    => 'inherit',
        'posts_per_page' => -1,
        'meta_query'     => [
            'relation' => 'AND',
            [
                'key'     => '_ewo_is_converted',
                'value'   => '1',
                'compare' => '=',
            ],
            [
                'key'     => '_ewo_original_deleted',
                'compare' => 'NOT EXISTS',
            ],
            [
                'key'     => '_ewo_conversion_timestamp',
                'value'   => $threshold,
                'compare' => '<=',
                'type'    => 'NUMERIC',
            ],
        ],
    ];

    $query = new WP_Query($args);
    $upload_dir = wp_get_upload_dir();

    foreach ($query->posts as $attachment) {
        $metadata = wp_get_attachment_metadata($attachment->ID);
        if (!$metadata || !isset($metadata['file'])) continue;

        // Eliminar todos los tamaños de la imagen original
        $image_sizes = isset($metadata['sizes']) ? $metadata['sizes'] : [];
        $image_sizes['full'] = ['file' => basename($metadata['file'])];

        foreach ($image_sizes as $size_data) {
            $file_path = $upload_dir['basedir'] . '/' . (dirname($metadata['file']) !== '.' ? dirname($metadata['file']) . '/' : '') . $size_data['file'];
            if (file_exists($file_path)) {
                @unlink($file_path);
            }
        }
        update_post_meta($attachment->ID, '_ewo_original_deleted', true);
    }
}

/**
 * Tarea semanal: escanea y convierte nuevas imágenes.
 */
function ewo_scheduled_scan() {
    $images_to_process = ewo_get_images_to_process();
    $quality = get_option('ewo_last_quality_used', 85); // Usar la última calidad o 85 por defecto

    foreach ($images_to_process as $image) {
        ewo_perform_conversion($image['id'], $quality);
    }
}

/**
 * Obtiene una lista de imágenes que necesitan ser procesadas.
 * @return array
 */
function ewo_get_images_to_process() {
    $args = [
        'post_type'      => 'attachment',
        'post_status'    => 'inherit',
        'posts_per_page' => -1,
        'post_mime_type' => ['image/jpeg', 'image/png', 'image/jpg'], // Añadido image/jpg por si acaso
    ];
    
    $query = new WP_Query($args);
    $images = [];

    foreach ($query->posts as $attachment) {
        // Comprobación manual en lugar de depender solo de meta_query
        if (!get_post_meta($attachment->ID, '_ewo_is_converted', true)) {
            $images[] = [
                'id'   => $attachment->ID,
                'file' => basename(get_attached_file($attachment->ID)),
            ];
        }
    }
    return $images;
}


/**
 * Lógica central para convertir una imagen.
 * @return true|WP_Error
 */
function ewo_perform_conversion($attachment_id, $quality) {
    $metadata = wp_get_attachment_metadata($attachment_id);
    if (!$metadata) {
        return new WP_Error('conversion_error', 'No se encontraron metadatos para el adjunto.');
    }
    
    $upload_dir = wp_get_upload_dir();
    $converted_files = 0;

    $image_sizes = isset($metadata['sizes']) ? $metadata['sizes'] : [];
    $image_sizes['full'] = ['file' => basename($metadata['file'])];

    foreach ($image_sizes as $size_data) {
        $file_path = $upload_dir['basedir'] . '/' . (dirname($metadata['file']) !== '.' ? dirname($metadata['file']) . '/' : '') . $size_data['file'];
        if (!file_exists($file_path)) continue;

        $webp_path = $file_path . '.webp';
        $mime_type = get_post_mime_type($attachment_id);
        $image = ($mime_type === 'image/png') ? @imagecreatefrompng($file_path) : @imagecreatefromjpeg($file_path);

        if ($image) {
            if ($mime_type === 'image/png') {
                imagepalettetotruecolor($image);
                imagealphablending($image, true);
                imagesavealpha($image, true);
            }
            if (@imagewebp($image, $webp_path, $quality)) {
                $converted_files++;
            }
            imagedestroy($image);
        }
    }

    if ($converted_files > 0) {
        global $wpdb;
        foreach ($image_sizes as $size_data) {
            $original_url = $upload_dir['baseurl'] . '/' . (dirname($metadata['file']) !== '.' ? dirname($metadata['file']) . '/' : '') . $size_data['file'];
            $webp_url = $original_url . '.webp';
            $wpdb->query($wpdb->prepare("UPDATE {$wpdb->posts} SET post_content = REPLACE(post_content, %s, %s)", $original_url, $webp_url));
            $wpdb->query($wpdb->prepare("UPDATE {$wpdb->postmeta} SET meta_value = REPLACE(meta_value, %s, %s)", $original_url, $webp_url));
        }
        update_post_meta($attachment_id, '_ewo_is_converted', true);
        update_post_meta($attachment_id, '_ewo_conversion_timestamp', time());
        update_option('ewo_last_quality_used', $quality); // Guardar la última calidad usada
        return true;
    }

    return new WP_Error('conversion_error', 'No se pudo convertir ningún archivo para este adjunto.');
}
